/*
 * Decompiled with CFR 0.152.
 */
package ghidra.server;

import ghidra.framework.remote.RepositoryChangeEvent;
import ghidra.framework.remote.RepositoryHandle;
import ghidra.framework.remote.User;
import ghidra.framework.store.FileSystem;
import ghidra.framework.store.FileSystemListener;
import ghidra.framework.store.local.IndexedLocalFileSystem;
import ghidra.framework.store.local.IndexedV1LocalFileSystem;
import ghidra.framework.store.local.LocalFileSystem;
import ghidra.framework.store.local.MangledLocalFileSystem;
import ghidra.framework.store.local.RepositoryLogger;
import ghidra.server.RepositoryManager;
import ghidra.server.remote.RepositoryHandleImpl;
import ghidra.server.store.RepositoryFile;
import ghidra.server.store.RepositoryFolder;
import ghidra.util.InvalidNameException;
import ghidra.util.NamingUtilities;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.UserAccessException;
import ghidra.util.timer.GTimer;
import ghidra.util.timer.GTimerMonitor;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class Repository
implements FileSystemListener,
RepositoryLogger {
    static final Logger log = LogManager.getLogger(Repository.class);
    private static final String READ_ONLY_STR = "READ_ONLY";
    private static final String WRITE_STR = "WRITE";
    private static final String ADMIN_STR = "ADMIN";
    private static final String ANONYMOUS_STR = "=ANONYMOUS_ALLOWED";
    private static final String[] TYPE_NAMES = new String[]{"READ_ONLY", "WRITE", "ADMIN"};
    private static final String INDEX_MIGRATION_MARKER_FILE = "~MIGRATE";
    private static final String ACCESS_CONTROL_FILENAME = "userAccess.acl";
    public static final User ANONYMOUS_USER = new User("-anonymous-", 0);
    private boolean valid;
    private GTimerMonitor clientCheckTimerMonitor;
    private RepositoryManager mgr;
    private LocalFileSystem fileSystem;
    private RepositoryFolder rootFolder;
    private String name;
    private File userAccessFile;
    private LinkedHashMap<String, User> userMap = new LinkedHashMap();
    private boolean anonymousAccessAllowed;
    private ArrayList<RepositoryHandleImpl> handleList = new ArrayList();
    private ArrayList<RepositoryChangeEvent> eventQueue = new ArrayList();
    private boolean dispatchSuspended = false;

    public Repository(RepositoryManager mgr, String currentUser, File rootFile, String name) throws IOException {
        this.mgr = mgr;
        this.name = name;
        log.info("Loading " + name + " ...");
        long t = System.currentTimeMillis();
        boolean create = rootFile.list().length == 0;
        boolean performMigration = !create && this.checkForIndexMigration(rootFile);
        this.fileSystem = LocalFileSystem.getLocalFileSystem((String)rootFile.getAbsolutePath(), (boolean)create, (boolean)true, (boolean)false, (boolean)false);
        if (performMigration) {
            this.fileSystem = this.performMigration(rootFile, this.fileSystem);
        }
        try {
            int count = this.fileSystem.getItemCount();
            log.info("   ... loading " + count + " files ...");
        }
        catch (UnsupportedOperationException count) {
            // empty catch block
        }
        this.fileSystem.setAssociatedRepositoryLogger((RepositoryLogger)this);
        this.fileSystem.addFileSystemListener((FileSystemListener)this);
        this.rootFolder = new RepositoryFolder(this, this.fileSystem);
        this.userAccessFile = new File(rootFile, ACCESS_CONTROL_FILENAME);
        if (currentUser != null) {
            this.userMap.put(currentUser, new User(currentUser, 2));
            this.writeUserList(currentUser, this.userMap, false);
        } else {
            this.readAccessFile();
        }
        this.valid = true;
        this.scheduleHandleCheck();
        log.info("   " + name + " load complete. " + (this.anonymousAccessAllowed() ? "(allows anonymous)" : ""));
        String loadTime = RepositoryManager.getElapsedTimeSince(t);
        if (loadTime != null) {
            log.info("   load time: " + loadTime);
        }
    }

    private boolean checkForIndexMigration(File rootFile) {
        File indexMigrationMarkerFile = new File(rootFile, INDEX_MIGRATION_MARKER_FILE);
        if (indexMigrationMarkerFile.exists()) {
            indexMigrationMarkerFile.delete();
            return true;
        }
        return false;
    }

    private LocalFileSystem performMigration(File rootFile, LocalFileSystem fs) throws IOException {
        if (fs instanceof IndexedV1LocalFileSystem) {
            return fs;
        }
        if (fs instanceof IndexedLocalFileSystem) {
            this.log(null, "Migrating repository to latest indexed filesystem version (V1)...", null);
            fs.dispose();
            IndexedV1LocalFileSystem.rebuild((File)rootFile);
        } else if (fs instanceof MangledLocalFileSystem) {
            this.log(null, "Migrating repository to indexed filesystem storage...", null);
            ((MangledLocalFileSystem)fs).convertToIndexedLocalFileSystem();
        } else {
            return fs;
        }
        return LocalFileSystem.getLocalFileSystem((String)rootFile.getAbsolutePath(), (boolean)false, (boolean)true, (boolean)false, (boolean)false);
    }

    private void scheduleHandleCheck() {
        this.clientCheckTimerMonitor = GTimer.scheduleRunnable((long)RepositoryHandle.CLIENT_CHECK_PERIOD, () -> {
            LocalFileSystem localFileSystem = this.fileSystem;
            synchronized (localFileSystem) {
                RepositoryHandleImpl[] handles = this.getHandles();
                for (int i = 0; i < handles.length; ++i) {
                    handles[i].checkHandle();
                }
                this.scheduleHandleCheck();
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void dispose() {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            RepositoryHandleImpl[] handles;
            if (this.clientCheckTimerMonitor != null) {
                this.clientCheckTimerMonitor.cancel();
                this.clientCheckTimerMonitor = null;
            }
            ArrayList<RepositoryHandleImpl> arrayList = this.handleList;
            synchronized (arrayList) {
                handles = new RepositoryHandleImpl[this.handleList.size()];
                this.handleList.toArray(handles);
            }
            for (int i = 0; i < handles.length; ++i) {
                handles[i].dispose();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void suspendEventDispatching() {
        ArrayList<RepositoryChangeEvent> arrayList = this.eventQueue;
        synchronized (arrayList) {
            this.dispatchSuspended = true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void sendChangeEvent(RepositoryChangeEvent event) {
        ArrayList<RepositoryChangeEvent> arrayList = this.eventQueue;
        synchronized (arrayList) {
            this.eventQueue.add(event);
        }
        if (!this.dispatchSuspended) {
            this.flushChangeEvents();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void flushChangeEvents() {
        RepositoryChangeEvent[] events;
        ArrayList<RepositoryChangeEvent> arrayList = this.eventQueue;
        synchronized (arrayList) {
            this.dispatchSuspended = false;
            if (this.eventQueue.isEmpty()) {
                return;
            }
            events = new RepositoryChangeEvent[this.eventQueue.size()];
            this.eventQueue.toArray(events);
            this.eventQueue.clear();
        }
        RepositoryHandleImpl[] handles = this.getHandles();
        for (int i = 0; i < handles.length; ++i) {
            handles[i].dispatchEvents(events);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private RepositoryHandleImpl[] getHandles() {
        RepositoryHandleImpl[] handles;
        ArrayList<RepositoryHandleImpl> arrayList = this.handleList;
        synchronized (arrayList) {
            handles = new RepositoryHandleImpl[this.handleList.size()];
            this.handleList.toArray(handles);
        }
        return handles;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addHandle(RepositoryHandleImpl handle) {
        ArrayList<RepositoryHandleImpl> arrayList = this.handleList;
        synchronized (arrayList) {
            this.handleList.add(handle);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void dropHandle(RepositoryHandleImpl handle) {
        ArrayList<RepositoryHandleImpl> arrayList = this.handleList;
        synchronized (arrayList) {
            this.handleList.remove(handle);
        }
    }

    public String getName() {
        return this.name;
    }

    public int getItemCount() throws IOException, UnsupportedOperationException {
        return this.fileSystem.getItemCount();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public RepositoryFolder getFolder(String currentUser, String folderPath, boolean create) throws InvalidNameException, IOException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            this.validate();
            if (!folderPath.startsWith(FileSystem.SEPARATOR)) {
                throw new IOException("Absolute path required");
            }
            RepositoryFolder folder = this.rootFolder;
            StringTokenizer st = new StringTokenizer(folderPath.substring(1), FileSystem.SEPARATOR);
            while (folder != null && st.hasMoreElements()) {
                String folderName = st.nextToken();
                RepositoryFolder next = folder.getFolder(folderName);
                if (next == null && create) {
                    next = folder.createFolder(folderName, currentUser);
                }
                folder = next;
            }
            return folder;
        }
    }

    public String[] getServerUserList(String currentUser) throws IOException {
        if ("-anonymous-".equals(currentUser)) {
            return new String[0];
        }
        return this.mgr.getUserManager().getUsers();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setUserList(String currentUser, User[] users, boolean allowAnonymousAccess) throws UserAccessException, IOException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            this.validate();
            this.validateAdminPrivilege(currentUser);
            LinkedHashMap<String, User> newUserMap = new LinkedHashMap<String, User>();
            for (int i = 0; i < users.length; ++i) {
                String userName = users[i].getName();
                if ("-anonymous-".equals(userName)) continue;
                newUserMap.put(userName, users[i]);
            }
            User user = (User)newUserMap.get(currentUser);
            if (user == null || !user.isAdmin()) {
                throw new UserAccessException("User may not remove or change permissions for self");
            }
            try {
                this.anonymousAccessAllowed = allowAnonymousAccess && this.mgr.anonymousAccessAllowed();
                this.writeUserList(currentUser, newUserMap, this.anonymousAccessAllowed);
                if (allowAnonymousAccess != this.anonymousAccessAllowed) {
                    this.log(null, "Enablement of Anonymous access setting ignored", currentUser);
                }
                this.userMap = newUserMap;
            }
            catch (FileNotFoundException e) {
                log.error("File not found for " + this.userAccessFile.getAbsolutePath());
            }
            catch (IOException e) {
                String msg = e.getMessage();
                if (msg == null) {
                    msg = e.toString();
                }
                log.error("Failed to write user access file: " + msg);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean setUserPermission(String username, int permission) throws IOException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            if (permission < 0 || permission > 2) {
                throw new IllegalArgumentException("Invalid permission: " + permission);
            }
            if (this.mgr.getUserManager().isValidUser(username)) {
                User newUser = new User(username, permission);
                User oldUser = this.userMap.put(username, newUser);
                this.writeUserList(this.userMap, this.anonymousAccessAllowed);
                if (oldUser != null) {
                    log.info("User access to repository '" + this.name + "' changed: " + String.valueOf(newUser));
                } else {
                    log.info("User access granted to repository '" + this.name + "': " + String.valueOf(newUser));
                }
                return true;
            }
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean removeUser(String username) throws IOException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            if (this.userMap.remove(username) != null) {
                this.writeUserList(this.userMap, this.anonymousAccessAllowed);
                log.info("User access removed from repository '" + this.name + "': " + username);
                return true;
            }
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public User[] getUserList(String currentUser) throws UserAccessException, IOException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            this.validate();
            if ("-anonymous-".equals(currentUser)) {
                return new User[0];
            }
            this.validateReadPrivilege(currentUser);
            User[] users = new User[this.userMap.size()];
            int i = 0;
            Iterator<User> iter = this.userMap.values().iterator();
            while (iter.hasNext()) {
                users[i++] = iter.next();
            }
            return users;
        }
    }

    public boolean anonymousAccessAllowed() {
        return this.anonymousAccessAllowed;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public User getUser(String username) {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            if (this.anonymousAccessAllowed && "-anonymous-".equals(username)) {
                return ANONYMOUS_USER;
            }
            User user = this.userMap.get(username);
            if (user == null && this.anonymousAccessAllowed) {
                return new User(username, 0);
            }
            return user;
        }
    }

    private void writeUserList(String currentUser, LinkedHashMap<String, User> newUserMap, boolean allowAnonymous) throws UserAccessException, IOException {
        User user = newUserMap.get(currentUser);
        if (user == null || !user.isAdmin()) {
            throw new UserAccessException(currentUser + " must have ADMIN privilege!");
        }
        this.writeUserList(newUserMap, allowAnonymous);
        log.info("User access list for repository '" + this.name + "' updated by: " + currentUser);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeUserList(LinkedHashMap<String, User> newUserMap, boolean allowAnonymous) throws IOException {
        File temp = new File(this.userAccessFile.getParentFile(), "tempAccess.tmp");
        temp.delete();
        try (PrintWriter out = new PrintWriter(new FileOutputStream(temp));){
            out.println(";");
            out.println("; User Access List for " + this.name + ": Auto-generated on " + String.valueOf(new Date()));
            out.println(";");
            if (allowAnonymous) {
                out.println(ANONYMOUS_STR);
            }
            for (User user : newUserMap.values()) {
                String line = user.getName() + "=" + TYPE_NAMES[user.getPermissionType()];
                out.println(line);
            }
            out.flush();
        }
        this.userAccessFile.delete();
        temp.renameTo(this.userAccessFile);
    }

    void delete(String currentUser) throws IOException, UserAccessException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            this.validate();
            this.validateAdminPrivilege(currentUser);
            throw new IOException("Delete repository not yet implemented");
        }
    }

    static String getFormattedUserPermissions(File repositoryDir, String pad) {
        StringBuilder buf = new StringBuilder();
        File userAccessFile = new File(repositoryDir, ACCESS_CONTROL_FILENAME);
        try {
            ArrayList<User> list = new ArrayList<User>();
            boolean anonymousAccessAllowed = Repository.readAccessFile(userAccessFile, list);
            Collections.sort(list);
            if (anonymousAccessAllowed) {
                buf.append(pad + "* Anonymous read-only access permitted *\n");
            }
            for (User user : list) {
                buf.append(pad + String.valueOf(user) + "\n");
            }
        }
        catch (IOException e) {
            System.out.println(pad + "Failed to read repository access file: " + e.getMessage());
        }
        return buf.toString();
    }

    static String getFormattedUserPermissions(File repositoryDir, String pad, Set<String> listUserAccess) {
        StringBuilder buf = null;
        File userAccessFile = new File(repositoryDir, ACCESS_CONTROL_FILENAME);
        try {
            ArrayList<User> list = new ArrayList<User>();
            Repository.readAccessFile(userAccessFile, list);
            Collections.sort(list);
            for (User user : list) {
                if (!listUserAccess.contains(user.getName())) continue;
                if (buf == null) {
                    buf = new StringBuilder();
                }
                buf.append(pad + String.valueOf(user) + "\n");
            }
        }
        catch (IOException e) {
            System.out.println(pad + "Failed to read repository access file: " + e.getMessage());
        }
        return buf != null ? buf.toString() : null;
    }

    private void readAccessFile() throws IOException {
        if (!this.userAccessFile.exists()) {
            return;
        }
        ArrayList<User> list = new ArrayList<User>();
        this.anonymousAccessAllowed = Repository.readAccessFile(this.userAccessFile, list) && this.mgr.anonymousAccessAllowed();
        LinkedHashMap<String, User> newUserMap = new LinkedHashMap<String, User>();
        Iterator<User> iter = list.iterator();
        boolean hasAdmin = false;
        while (iter.hasNext()) {
            User user = iter.next();
            hasAdmin |= user.isAdmin();
            newUserMap.put(user.getName(), user);
        }
        if (!hasAdmin) {
            throw new IOException("Repository does not have an Admin");
        }
        this.userMap = newUserMap;
    }

    private static boolean readAccessFile(File userAccessFile, List<User> users) throws IOException {
        boolean allowAnonymous = false;
        try (BufferedReader reader = new BufferedReader(new FileReader(userAccessFile));){
            String line = "";
            while ((line = reader.readLine()) != null) {
                if (line.startsWith(";")) continue;
                if (ANONYMOUS_STR.equals(line = line.trim())) {
                    allowAnonymous = true;
                    continue;
                }
                User user = Repository.processAccessLine(line);
                if (user == null) continue;
                users.add(user);
            }
        }
        return allowAnonymous;
    }

    private static User processAccessLine(String line) {
        int pos = line.indexOf(61);
        if (pos > 0) {
            String userName = line.substring(0, pos).trim();
            String typeName = line.substring(pos + 1).trim();
            for (int i = 0; i < TYPE_NAMES.length; ++i) {
                if (!typeName.equals(TYPE_NAMES[i])) continue;
                return new User(userName, i);
            }
        }
        return null;
    }

    public void validate() throws IOException {
        if (!this.valid) {
            throw new IOException("Repository has been deleted");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public User validateAdminPrivilege(String currentUser) throws UserAccessException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            User user = this.getUser(currentUser);
            if (user == null) {
                throw new UserAccessException("User " + currentUser + " was not found in the '" + this.name + "' repository access list.");
            }
            if (!user.isAdmin()) {
                throw new UserAccessException("User " + currentUser + " does not have Admin privilege.");
            }
            return user;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public User validateWritePrivilege(String currentUser) throws UserAccessException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            User user = this.getUser(currentUser);
            if (user == null) {
                throw new UserAccessException("User " + currentUser + " was not found in the '" + this.name + "' repository access list.");
            }
            if (user.isReadOnly()) {
                throw new UserAccessException("User " + currentUser + " does not have write privilege.");
            }
            return user;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public User validateReadPrivilege(String currentUser) throws UserAccessException {
        LocalFileSystem localFileSystem = this.fileSystem;
        synchronized (localFileSystem) {
            User user = this.getUser(currentUser);
            if (user == null) {
                throw new UserAccessException("User " + currentUser + " was not found in the '" + this.name + "' repository access list.");
            }
            return user;
        }
    }

    public void folderCreated(String parentPath, String folderName) {
        try {
            RepositoryFolder folder = this.getFolder(null, parentPath, false);
            if (folder == null || folder.getFolder(folderName) == null) {
                RepositoryManager.log(this.name, RepositoryFolder.makePathname(parentPath, folderName), "ERROR! folder not found", null);
                return;
            }
        }
        catch (InvalidNameException e) {
            throw new AssertException();
        }
        catch (IOException e) {
            RepositoryManager.log(this.name, RepositoryFolder.makePathname(parentPath, folderName), "ERROR! " + e.getMessage(), null);
        }
        RepositoryChangeEvent event = new RepositoryChangeEvent(0, parentPath, folderName, null, null);
        this.sendChangeEvent(event);
    }

    public void itemCreated(String parentPath, String itemName) {
        try {
            RepositoryFolder folder = this.getFolder(null, parentPath, false);
            if (folder == null || folder.getFile(itemName) == null) {
                RepositoryManager.log(this.name, RepositoryFolder.makePathname(parentPath, itemName), "file not found", null);
                return;
            }
        }
        catch (InvalidNameException e) {
            throw new AssertException();
        }
        catch (IOException e) {
            RepositoryManager.log(this.name, RepositoryFolder.makePathname(parentPath, itemName), "ERROR! " + e.getMessage(), null);
        }
        RepositoryChangeEvent event = new RepositoryChangeEvent(1, parentPath, itemName, null, null);
        this.sendChangeEvent(event);
    }

    public void folderDeleted(String parentPath, String folderName) {
        try {
            RepositoryFolder folder = this.getFolder(null, RepositoryFolder.makePathname(parentPath, folderName), false);
            if (folder != null) {
                folder.delete();
            }
        }
        catch (InvalidNameException e) {
            throw new AssertException();
        }
        catch (IOException e) {
            RepositoryManager.log(this.name, parentPath, "ERROR! " + e.getMessage(), null);
        }
        RepositoryChangeEvent event = new RepositoryChangeEvent(2, parentPath, folderName, null, null);
        this.sendChangeEvent(event);
    }

    public void folderMoved(String parentPath, String folderName, String newParentPath) {
        RepositoryChangeEvent event = new RepositoryChangeEvent(3, parentPath, folderName, newParentPath, null);
        this.sendChangeEvent(event);
    }

    public void folderRenamed(String parentPath, String oldFolderName, String newFolderName) {
        RepositoryChangeEvent event = new RepositoryChangeEvent(4, parentPath, oldFolderName, null, newFolderName);
        this.sendChangeEvent(event);
    }

    public void itemDeleted(String parentPath, String itemName) {
        RepositoryChangeEvent event = new RepositoryChangeEvent(5, parentPath, itemName, null, null);
        this.sendChangeEvent(event);
    }

    public void itemRenamed(String parentPath, String oldItemName, String newItemName) {
        RepositoryChangeEvent event = new RepositoryChangeEvent(6, parentPath, oldItemName, null, newItemName);
        this.sendChangeEvent(event);
    }

    public void itemMoved(String parentPath, String itemName, String newParentPath, String newName) {
        RepositoryChangeEvent event = new RepositoryChangeEvent(7, parentPath, itemName, newParentPath, newName);
        this.sendChangeEvent(event);
    }

    public void itemChanged(String parentPath, String itemName) {
        boolean syncErr = true;
        try {
            RepositoryFile rf;
            RepositoryFolder parentFolder = this.getFolder(null, parentPath, false);
            if (parentFolder != null && (rf = parentFolder.getFile(itemName)) != null) {
                rf.itemChanged();
                syncErr = false;
            }
        }
        catch (InvalidNameException e) {
            throw new AssertException();
        }
        catch (IOException e) {
            RepositoryManager.log(this.name, RepositoryFolder.makePathname(parentPath, itemName), "ERROR! " + e.getMessage(), null);
        }
        if (syncErr) {
            RepositoryManager.log(this.name, null, "ERROR! Repository instance may be out-of-sync", null);
            return;
        }
        RepositoryChangeEvent event = new RepositoryChangeEvent(8, parentPath, itemName, null, null);
        this.sendChangeEvent(event);
    }

    public void syncronize() {
    }

    public void log(String path, String msg, String user) {
        RepositoryManager.log(this.name, path, msg, user);
    }

    static boolean markRepositoryForIndexMigration(File serverDir, String repositoryName, boolean silent) {
        File repoDir = new File(serverDir, NamingUtilities.mangle((String)repositoryName));
        if (!repoDir.isDirectory() || repositoryName.startsWith(LocalFileSystem.HIDDEN_DIR_PREFIX)) {
            System.err.println("Repository '" + repositoryName + "' not found");
            return false;
        }
        String rootPath = repoDir.getAbsolutePath();
        boolean isIndexed = IndexedLocalFileSystem.isIndexed((String)rootPath);
        if (isIndexed) {
            try {
                int indexVersion = IndexedLocalFileSystem.readIndexVersion((String)rootPath);
                if (indexVersion >= 1) {
                    if (!silent) {
                        System.err.println("Repository '" + repositoryName + "' is already indexed!");
                    }
                    return false;
                }
            }
            catch (IOException e) {
                System.err.println("Repository access error (" + repositoryName + "): " + e.getMessage());
                return false;
            }
        }
        File indexMigrationMarkerFile = new File(repoDir, INDEX_MIGRATION_MARKER_FILE);
        try {
            indexMigrationMarkerFile.createNewFile();
            System.out.println("Repository '" + repositoryName + "' marked for index migration on server restart");
        }
        catch (IOException e) {
            System.err.println("Failed to mark repository for migration: " + e.getMessage());
        }
        return true;
    }

    public Object getSyncObject() {
        return this.fileSystem;
    }
}

