/*
 * Decompiled with CFR 0.152.
 */
package ghidra.file.formats.ios;

import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.ByteArrayProvider;
import ghidra.app.util.bin.ByteProvider;
import ghidra.app.util.bin.format.macho.MachHeader;
import ghidra.app.util.bin.format.macho.Section;
import ghidra.app.util.bin.format.macho.commands.DyldInfoCommand;
import ghidra.app.util.bin.format.macho.commands.DynamicSymbolTableCommand;
import ghidra.app.util.bin.format.macho.commands.LinkEditDataCommand;
import ghidra.app.util.bin.format.macho.commands.LoadCommand;
import ghidra.app.util.bin.format.macho.commands.NList;
import ghidra.app.util.bin.format.macho.commands.SegmentCommand;
import ghidra.app.util.bin.format.macho.commands.SymbolTableCommand;
import ghidra.formats.gfilesystem.FSRL;
import ghidra.util.DataConverter;
import ghidra.util.LittleEndianDataConverter;
import ghidra.util.Msg;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.NotFoundException;
import ghidra.util.task.TaskMonitor;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ExtractedMacho {
    protected ByteProvider provider;
    protected long providerOffset;
    protected byte[] footer;
    protected BinaryReader reader;
    protected MachHeader machoHeader;
    protected SegmentCommand textSegment;
    protected SegmentCommand linkEditSegment;
    protected Map<SegmentCommand, Integer> packedSegmentStarts = new HashMap<SegmentCommand, Integer>();
    protected Map<SegmentCommand, Integer> packedSegmentAdjustments = new HashMap<SegmentCommand, Integer>();
    protected Map<LoadCommand, Integer> packedLinkEditDataStarts = new HashMap<LoadCommand, Integer>();
    protected byte[] packed;
    protected TaskMonitor monitor;

    public ExtractedMacho(ByteProvider provider, long providerOffset, MachHeader machoHeader, byte[] footer, TaskMonitor monitor) throws IOException, CancelledException {
        this.provider = provider;
        this.providerOffset = providerOffset;
        this.footer = footer;
        this.machoHeader = machoHeader;
        this.textSegment = machoHeader.getSegment("__TEXT");
        this.linkEditSegment = machoHeader.getSegment("__LINKEDIT");
        this.reader = new BinaryReader(provider, machoHeader.isLittleEndian());
        this.monitor = monitor;
    }

    public void pack() throws IOException, CancelledException {
        int packedSize = 0;
        int packedLinkEditSize = 0;
        for (SegmentCommand segment : this.machoHeader.getAllSegments()) {
            this.monitor.checkCancelled();
            this.packedSegmentStarts.put(segment, packedSize);
            if (segment == this.linkEditSegment) {
                for (LoadCommand cmd : this.machoHeader.getLoadCommands()) {
                    if (cmd instanceof SymbolTableCommand) {
                        SymbolTableCommand symbolTable = (SymbolTableCommand)cmd;
                        symbolTable.addSymbols(this.getExtraSymbols());
                    }
                    int offset = cmd.getLinkerDataOffset();
                    int size = cmd.getLinkerDataSize();
                    if (offset == 0 || size == 0) continue;
                    this.packedLinkEditDataStarts.put(cmd, packedLinkEditSize);
                    packedLinkEditSize += size;
                }
                packedSize += packedLinkEditSize;
                segment.setFileSize((long)packedLinkEditSize);
                segment.setVMsize((long)packedLinkEditSize);
            } else {
                packedSize = (int)((long)packedSize + segment.getFileSize());
            }
            if (segment != this.textSegment || segment.getFileOffset() != 0L) continue;
            segment.setFileOffset(this.providerOffset);
            this.packedSegmentAdjustments.put(segment, (int)this.providerOffset);
        }
        this.packed = new byte[packedSize += this.footer.length];
        for (SegmentCommand segment : this.machoHeader.getAllSegments()) {
            byte[] bytes;
            this.monitor.checkCancelled();
            long segmentSize = segment.getFileSize();
            ByteProvider segmentProvider = this.getSegmentProvider(segment);
            if (segment.getFileOffset() + segmentSize > segmentProvider.length()) {
                segmentSize = segmentProvider.length() - segment.getFileOffset();
                Msg.warn((Object)this, (Object)(segment.getSegmentName() + " segment extends beyond end of file.  Truncating..."));
            }
            if (segment == this.linkEditSegment) {
                bytes = this.createPackedLinkEditSegment(segmentProvider, packedLinkEditSize);
                this.adjustLinkEditAddress();
            } else {
                bytes = segmentProvider.readBytes(segment.getFileOffset(), segmentSize);
            }
            System.arraycopy(bytes, 0, this.packed, this.packedSegmentStarts.get(segment), bytes.length);
        }
        this.fixupLoadCommands();
        System.arraycopy(this.footer, 0, this.packed, this.packed.length - this.footer.length, this.footer.length);
    }

    public ByteProvider getByteProvider(FSRL fsrl) {
        return new ByteArrayProvider(this.packed, fsrl);
    }

    protected ByteProvider getSegmentProvider(SegmentCommand segment) throws IOException {
        return this.provider;
    }

    protected List<NList> getExtraSymbols() {
        return List.of();
    }

    protected long getPackedOffset(long fileOffset, SegmentCommand segment) throws NotFoundException {
        Integer segmentStart = this.packedSegmentStarts.get(segment);
        if (segmentStart != null) {
            return fileOffset - segment.getFileOffset() + (long)segmentStart.intValue();
        }
        throw new NotFoundException("Failed to convert Mach-O file offset to packed offset: 0x%x".formatted(fileOffset));
    }

    private byte[] createPackedLinkEditSegment(ByteProvider linkEditSegmentProvider, int packedLinkEditSize) throws IOException {
        byte[] packedLinkEdit = new byte[packedLinkEditSize];
        for (LoadCommand cmd : this.packedLinkEditDataStarts.keySet()) {
            SymbolTableCommand symbolTable;
            if (cmd instanceof SymbolTableCommand && (symbolTable = (SymbolTableCommand)cmd).getNumberOfSymbols() > 0) {
                int stringIndex;
                List symbols = symbolTable.getSymbols();
                byte[] packedSymbolStringTable = new byte[NList.getSize((List)symbols)];
                int nlistIndex = 0;
                int stringIndexOrig = stringIndex = ((NList)symbols.get(0)).getSize() * symbols.size();
                for (NList nlist : symbols) {
                    byte[] nlistArray = this.nlistToArray(nlist, stringIndex - stringIndexOrig);
                    byte[] stringArray = nlist.getString().getBytes(StandardCharsets.US_ASCII);
                    System.arraycopy(nlistArray, 0, packedSymbolStringTable, nlistIndex, nlistArray.length);
                    System.arraycopy(stringArray, 0, packedSymbolStringTable, stringIndex, stringArray.length);
                    nlistIndex += nlistArray.length;
                    stringIndex += stringArray.length + 1;
                }
                System.arraycopy(packedSymbolStringTable, 0, packedLinkEdit, this.packedLinkEditDataStarts.get(cmd), packedSymbolStringTable.length);
                continue;
            }
            byte[] bytes = linkEditSegmentProvider.readBytes((long)cmd.getLinkerDataOffset(), (long)cmd.getLinkerDataSize());
            System.arraycopy(bytes, 0, packedLinkEdit, this.packedLinkEditDataStarts.get(cmd), bytes.length);
        }
        return packedLinkEdit;
    }

    private byte[] nlistToArray(NList nlist, int stringIndex) {
        byte[] ret = new byte[nlist.getSize()];
        DataConverter conv = DataConverter.getInstance((!this.machoHeader.isLittleEndian() ? 1 : 0) != 0);
        conv.putInt(ret, 0, stringIndex);
        ret[4] = nlist.getType();
        ret[5] = nlist.getSection();
        conv.putShort(ret, 6, nlist.getDescription());
        if (nlist.is32bit()) {
            conv.putInt(ret, 8, (int)nlist.getValue());
        } else {
            conv.putLong(ret, 8, nlist.getValue());
        }
        return ret;
    }

    private void fixupLoadCommands() throws IOException {
        for (LoadCommand cmd : this.machoHeader.getLoadCommands()) {
            if (this.monitor.isCancelled()) break;
            switch (cmd.getCommandType()) {
                case 1: {
                    this.fixupSegment((SegmentCommand)cmd, false);
                    break;
                }
                case 25: {
                    this.fixupSegment((SegmentCommand)cmd, true);
                    break;
                }
                case 2: {
                    this.fixupSymbolTable((SymbolTableCommand)cmd);
                    break;
                }
                case 11: {
                    this.fixupDynamicSymbolTable((DynamicSymbolTableCommand)cmd);
                    break;
                }
                case -2147483614: 
                case 34: {
                    this.fixupDyldInfo((DyldInfoCommand)cmd);
                    break;
                }
                case -2147483597: 
                case -2147483596: 
                case 29: 
                case 30: 
                case 38: 
                case 41: 
                case 43: 
                case 46: {
                    this.fixupLinkEditData((LinkEditDataCommand)cmd);
                }
            }
        }
    }

    private void fixupSegment(SegmentCommand segment, boolean is64bit) throws IOException {
        long adjustment = this.packedSegmentAdjustments.getOrDefault(segment, 0).intValue();
        this.set(segment.getStartIndex() + (long)(is64bit ? 24 : 24), segment.getVMaddress(), is64bit ? 8 : 4);
        this.set(segment.getStartIndex() + (long)(is64bit ? 32 : 28), segment.getVMsize(), is64bit ? 8 : 4);
        this.fixup(segment.getStartIndex() + (long)(is64bit ? 40 : 32), adjustment, is64bit ? 8 : 4, segment);
        this.set(segment.getStartIndex() + (long)(is64bit ? 48 : 36), segment.getFileSize(), is64bit ? 8 : 4);
        long sectionStartIndex = segment.getStartIndex() + (long)(is64bit ? 72 : 56);
        for (Section section : segment.getSections()) {
            if (this.monitor.isCancelled()) break;
            if (section.getOffset() > 0 && section.getSize() > 0L) {
                this.fixup(sectionStartIndex + (long)(is64bit ? 48 : 40), adjustment, 4, segment);
            }
            if (section.getRelocationOffset() > 0) {
                this.fixup(sectionStartIndex + (long)(is64bit ? 56 : 48), adjustment, 4, segment);
            }
            sectionStartIndex += is64bit ? 80L : 68L;
        }
    }

    private void fixupSymbolTable(SymbolTableCommand cmd) throws IOException {
        if (cmd.getSymbolOffset() > 0) {
            long symbolOffset = this.fixup(cmd.getStartIndex() + 8L, this.getLinkEditAdjustment((LoadCommand)cmd), 4, this.linkEditSegment);
            this.set(cmd.getStartIndex() + 12L, cmd.getNumberOfSymbols(), 4);
            if (cmd.getStringTableOffset() > 0) {
                if (cmd.getNumberOfSymbols() > 0) {
                    this.set(cmd.getStartIndex() + 16L, symbolOffset + (long)(cmd.getNumberOfSymbols() * cmd.getSymbolAt(0).getSize()), 4);
                    this.set(cmd.getStartIndex() + 20L, cmd.getStringTableSize(), 4);
                } else {
                    this.set(cmd.getStartIndex() + 16L, symbolOffset, 4);
                    this.set(cmd.getStartIndex() + 20L, 0L, 4);
                }
            }
        }
    }

    private void fixupDynamicSymbolTable(DynamicSymbolTableCommand cmd) throws IOException {
        long adjustment = this.getLinkEditAdjustment((LoadCommand)cmd);
        if (cmd.getTableOfContentsOffset() > 0) {
            this.set(cmd.getStartIndex() + 32L, 0L, 8);
        }
        if (cmd.getModuleTableOffset() > 0) {
            this.set(cmd.getStartIndex() + 40L, 0L, 8);
        }
        if (cmd.getReferencedSymbolTableOffset() > 0) {
            this.set(cmd.getStartIndex() + 48L, 0L, 8);
        }
        if (cmd.getIndirectSymbolTableOffset() > 0) {
            this.fixup(cmd.getStartIndex() + 56L, adjustment, 4, this.linkEditSegment);
        }
        if (cmd.getExternalRelocationOffset() > 0) {
            this.set(cmd.getStartIndex() + 64L, 0L, 8);
        }
        if (cmd.getLocalRelocationOffset() > 0) {
            this.set(cmd.getStartIndex() + 72L, 0L, 8);
        }
    }

    private void fixupDyldInfo(DyldInfoCommand cmd) throws IOException {
        if (cmd.getRebaseOffset() > 0) {
            this.set(cmd.getStartIndex() + 8L, 0L, 8);
        }
        if (cmd.getBindOffset() > 0) {
            this.set(cmd.getStartIndex() + 16L, 0L, 8);
        }
        if (cmd.getWeakBindOffset() > 0) {
            this.set(cmd.getStartIndex() + 24L, 0L, 8);
        }
        if (cmd.getLazyBindOffset() > 0) {
            this.set(cmd.getStartIndex() + 32L, 0L, 8);
        }
        if (cmd.getExportOffset() > 0) {
            this.set(cmd.getStartIndex() + 40L, 0L, 8);
        }
    }

    private void fixupLinkEditData(LinkEditDataCommand cmd) throws IOException {
        if (cmd.getLinkerDataOffset() > 0) {
            this.fixup(cmd.getStartIndex() + 8L, this.getLinkEditAdjustment((LoadCommand)cmd), 4, this.linkEditSegment);
        }
    }

    private long getLinkEditAdjustment(LoadCommand cmd) {
        return (long)this.packedLinkEditDataStarts.getOrDefault(cmd, 0).intValue() - ((long)cmd.getLinkerDataOffset() - this.linkEditSegment.getFileOffset());
    }

    private void set(long fileOffset, long value, int size) throws IOException, IllegalArgumentException {
        if (size != 4 && size != 8) {
            throw new IllegalArgumentException("Size must be 4 or 8 (got " + size + ")");
        }
        try {
            byte[] newBytes = ExtractedMacho.toBytes(value, size);
            System.arraycopy(newBytes, 0, this.packed, (int)this.getPackedOffset(fileOffset, this.textSegment), newBytes.length);
        }
        catch (NotFoundException e) {
            Msg.warn((Object)this, (Object)e.getMessage());
        }
    }

    private long fixup(long fileOffset, long adjustment, int size, SegmentCommand segment) throws IOException, IllegalArgumentException {
        long value;
        if (size != 4 && size != 8) {
            throw new IllegalArgumentException("Size must be 4 or 8 (got " + size + ")");
        }
        long ret = value = this.reader.readUnsignedValue(fileOffset, size);
        value += adjustment;
        try {
            ret = this.getPackedOffset(value, segment);
            byte[] newBytes = ExtractedMacho.toBytes(ret, size);
            System.arraycopy(newBytes, 0, this.packed, (int)this.getPackedOffset(fileOffset, this.textSegment), newBytes.length);
        }
        catch (NotFoundException e) {
            Msg.warn((Object)this, (Object)e.getMessage());
        }
        return ret;
    }

    private void adjustLinkEditAddress() {
        if (this.machoHeader.is32bit()) {
            return;
        }
        this.linkEditSegment.setVMaddress(this.textSegment.getVMaddress() << 4);
    }

    public static byte[] toBytes(long value, int size) throws IllegalArgumentException {
        if (size != 4 && size != 8) {
            throw new IllegalArgumentException("Size must be 4 or 8 (got " + size + ")");
        }
        LittleEndianDataConverter converter = LittleEndianDataConverter.INSTANCE;
        return size == 8 ? converter.getBytes(value) : converter.getBytes((int)value);
    }
}

