/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.util.demangler.gnu;

import generic.jar.ResourceFile;
import generic.json.Json;
import ghidra.app.util.SymbolPathParser;
import ghidra.app.util.demangler.Demangled;
import ghidra.app.util.demangler.DemangledAddressTable;
import ghidra.app.util.demangler.DemangledDataType;
import ghidra.app.util.demangler.DemangledFunction;
import ghidra.app.util.demangler.DemangledFunctionPointer;
import ghidra.app.util.demangler.DemangledLambda;
import ghidra.app.util.demangler.DemangledObject;
import ghidra.app.util.demangler.DemangledParameter;
import ghidra.app.util.demangler.DemangledString;
import ghidra.app.util.demangler.DemangledTemplate;
import ghidra.app.util.demangler.DemangledThunk;
import ghidra.app.util.demangler.DemangledType;
import ghidra.app.util.demangler.DemangledVariable;
import ghidra.app.util.demangler.gnu.DemanglerParseException;
import ghidra.app.util.demangler.gnu.GnuDemanglerReplacement;
import ghidra.framework.Application;
import ghidra.util.Msg;
import ghidra.util.StringUtilities;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import utilities.util.FileUtilities;

public class GnuDemanglerParser {
    private static final String REPLACEMENT_FILE_EXTENSION = ".gnu.demangler.replacements.txt";
    private static final String CONSTRUCTION_VTABLE_FOR = "construction vtable for ";
    private static final String VTT_FOR = "VTT for ";
    private static final String VTABLE_FOR = "vtable for ";
    private static final String TYPEINFO_NAME_FOR = "typeinfo name for ";
    private static final String TYPEINFO_FN_FOR = "typeinfo fn for ";
    private static final String TYPEINFO_FOR = "typeinfo for ";
    private static final String COVARIANT_RETURN_THUNK = "covariant return thunk";
    private static final Set<String> ADDRESS_TABLE_PREFIXES = Set.of("construction vtable for ", "VTT for ", "vtable for ", "typeinfo fn for ", "typeinfo for ");
    private static final String OPERATOR = "operator";
    private static final String LAMBDA = "lambda";
    private static final String LAMBDA_START = "{lambda";
    private static final String VAR_ARGS = "...";
    private static final String THUNK = "thunk";
    private static final String CONST = " const";
    private static final char NULL_CHAR = '\u0000';
    private static final Pattern UNNECESSARY_PARENS_PATTERN = Pattern.compile("\\s*(?:const){0,1}\\((.*)\\)\\s*");
    private static final Pattern VARARGS_IN_PARENS = Pattern.compile("\\((.*)\\)" + Pattern.quote("..."));
    private static final Pattern ARRAY_POINTER_REFERENCE_PATTERN = Pattern.compile("\\s(?:const[\\[\\]\\d\\*&]{0,4}\\s)*\\(([&*])\\)\\s*((?:\\[.*?\\])+)");
    private static final Pattern ARRAY_POINTER_REFERENCE_PIECE_PATTERN = Pattern.compile("\\([&*]\\)\\s*\\[.*?\\]");
    private static final Pattern CAST_PATTERN = Pattern.compile("\\((?:\\w+\\s)*\\w+(?:::\\w+)*\\)\\s*-{0,1}\\w+");
    private static final Pattern OVERLOAD_OPERATOR_NAME_PATTERN = GnuDemanglerParser.createOverloadedOperatorNamePattern();
    private static final Pattern CONVERSION_OPERATOR_PATTERN = Pattern.compile("(.*operator) (.*)\\(\\).*");
    private static final Pattern NEW_DELETE_OPERATOR_PATTERN = Pattern.compile("(.*operator) (new|delete)(\\[\\])?\\((.*)\\).*");
    private static final Pattern LAMBDA_PATTERN = Pattern.compile(".*(\\{lambda\\((.*)\\)(#\\d+)\\})(.*)");
    private static final Pattern UNNAMED_TYPE_PATTERN = Pattern.compile("(\\{unnamed type#\\d+})");
    private static final Pattern ARRAY_DATA_PATTERN = Pattern.compile("(.+)\\{(.+)\\[\\d*\\]\\{.*\\}\\}");
    private static final Pattern DESCRIPTIVE_PREFIX_PATTERN = Pattern.compile("((.+ )(for|to) )(.+)");
    private static final Pattern GLOBAL_CTOR_DTOR_FOR_PATTERN = Pattern.compile("(global (constructors|destructors) keyed to )(.+)");
    private static final Pattern DECLTYPE_RETURN_TYPE_PATTERN = Pattern.compile("decltype \\(.*\\)");
    private static final Pattern LITERAL_NUMBER_PATTERN = Pattern.compile("-*\\d+[ul]{0,1}");
    private static final Pattern RUST_LEGACY_SUFFIX_PATTERN = Pattern.compile("(.*)::h[0-9a-f]{16}");
    private static final List<GnuDemanglerReplacement> REPLACEMENTS = GnuDemanglerParser.loadStandardReplacements();
    private String mangledSource;
    private String demangledSource;

    private static Pattern createOverloadedOperatorNamePattern() {
        String userDefinedLiteral;
        LinkedList<String> operators = new LinkedList<String>(List.of("++", "--", ">>=", "<<=", "->*", "->", "==", "!=", ">=", "<=", "&&", "||", ">>", "<<", "+=", "-=", "*=", "/=", "%=", "&=", "|=", "^=", "+", "-", "*", "/", "%", "~", "^", "&", "|", "!", "<", ">", "=", ",", "()"));
        CollectionUtils.transform(operators, Pattern::quote);
        Object alternated = StringUtils.join(operators, (String)"|");
        String extra = userDefinedLiteral = "\"\"\\s_.+";
        alternated = (String)alternated + "|" + extra;
        String operatorTemplates = "(<.+>){0,1}";
        String operatorPrefix = "(.*operator(" + (String)alternated + ")\\s*" + operatorTemplates + ")\\s*";
        String parameters = "(\\(.*\\))";
        String trailing = "(.*)";
        return Pattern.compile(operatorPrefix + parameters + trailing);
    }

    public DemangledObject parse(String mangled, String demangled) throws DemanglerParseException {
        return this.parse(mangled, demangled, true);
    }

    public DemangledObject parse(String mangled, String demangled, boolean simplify) throws DemanglerParseException {
        demangled = this.cleanupRustLegacySymbol(demangled);
        if (simplify) {
            demangled = this.replaceStdLibraryTypes(demangled);
        }
        this.mangledSource = mangled;
        this.demangledSource = demangled;
        DemangledObjectBuilder builder = this.getSpecializedBuilder(demangled);
        if (builder != null) {
            return builder.build();
        }
        return this.parseFunctionOrVariable(demangled);
    }

    private DemangledObjectBuilder getSpecializedBuilder(String demangled) {
        SpecialPrefixHandler handler = this.getSpecialPrefixHandler(demangled);
        if (handler != null) {
            return handler;
        }
        OperatorHandler operatorHandler = this.getOperatorHandler(demangled);
        if (operatorHandler != null) {
            return operatorHandler;
        }
        if (this.mangledSource.startsWith("_ZZ") || this.mangledSource.startsWith("__ZZ")) {
            return new ItemInNamespaceHandler(demangled);
        }
        return null;
    }

    private OperatorHandler getOperatorHandler(String demangled) {
        OperatorHandler handler = new OverloadOperatorHandler(demangled);
        if (handler.matches(demangled)) {
            return handler;
        }
        handler = new ConversionOperatorHandler(demangled);
        if (handler.matches(demangled)) {
            return handler;
        }
        handler = new NewOrDeleteOperatorHandler(demangled);
        if (handler.matches(demangled)) {
            return handler;
        }
        return null;
    }

    private SpecialPrefixHandler getSpecialPrefixHandler(String demangled) {
        Matcher matcher = DESCRIPTIVE_PREFIX_PATTERN.matcher(demangled);
        if (matcher.matches()) {
            String prefix = matcher.group(1);
            String type = matcher.group(4);
            if (prefix.contains(THUNK)) {
                return new ThunkHandler(demangled, prefix, type);
            }
            if (ADDRESS_TABLE_PREFIXES.contains(prefix)) {
                return new AddressTableHandler(demangled, prefix, type);
            }
            if (prefix.startsWith(TYPEINFO_NAME_FOR)) {
                return new TypeInfoNameHandler(demangled, TYPEINFO_NAME_FOR);
            }
            Matcher arrayMatcher = ARRAY_DATA_PATTERN.matcher(type);
            if (arrayMatcher.matches()) {
                return new ArrayHandler(demangled, prefix, type);
            }
            Matcher globalCtorMatcher = GLOBAL_CTOR_DTOR_FOR_PATTERN.matcher(demangled);
            if (globalCtorMatcher.matches()) {
                prefix = globalCtorMatcher.group(1);
                type = globalCtorMatcher.group(3);
                return new GlobalCtorDtorHandler(demangled, prefix, type);
            }
            return new ItemInNamespaceHandler(demangled, prefix, type);
        }
        return null;
    }

    private DemangledObject parseFunctionOrVariable(String demangled) {
        FunctionSignatureParts signatureParts = new FunctionSignatureParts(this, demangled);
        if (!signatureParts.isValidFunction()) {
            return this.parseVariable(demangled);
        }
        DemangledFunction function = new DemangledFunction(this.mangledSource, demangled, null);
        String simpleName = signatureParts.getName();
        if (simpleName.endsWith(LAMBDA_START)) {
            String prefix = signatureParts.getRawParameterPrefix();
            int lambdaStart = prefix.length() - LAMBDA_START.length();
            String lambdaText = demangled.substring(lambdaStart);
            LambdaName lambdaName = this.getLambdaName(lambdaText);
            String uniqueName = lambdaName.getFullText();
            String escapedLambda = this.removeBadSpaces(uniqueName);
            simpleName = simpleName.replace(LAMBDA_START, escapedLambda);
            function = new DemangledLambda(this.mangledSource, demangled, null);
            function.setBackupPlateComment(lambdaName.getFullText());
        }
        this.setNameAndNamespace((DemangledObject)function, simpleName);
        for (DemangledParameter parameter : signatureParts.getParameters()) {
            function.addParameter(parameter);
        }
        String returnType = signatureParts.getReturnType();
        this.setReturnType(demangled, function, returnType);
        if (demangled.endsWith(CONST)) {
            function.setConst(true);
        }
        return function;
    }

    private String cleanupRustLegacySymbol(String demangled) {
        Matcher m = RUST_LEGACY_SUFFIX_PATTERN.matcher(demangled);
        if (m.matches()) {
            return m.group(1);
        }
        return demangled;
    }

    private String replaceStdLibraryTypes(String demangled) {
        String d = demangled;
        for (GnuDemanglerReplacement replacement : REPLACEMENTS) {
            d = StringUtils.replace((String)d, (String)replacement.find(), (String)replacement.replace());
        }
        return d;
    }

    private void setReturnType(String demangled, DemangledFunction function, String returnType) {
        String updatedReturnType = returnType;
        if (returnType != null && DECLTYPE_RETURN_TYPE_PATTERN.matcher(returnType).matches()) {
            updatedReturnType = null;
        }
        if (updatedReturnType != null) {
            function.setReturnType(this.parseReturnType(updatedReturnType));
            return;
        }
        DemangledDataType defaultReturnType = new DemangledDataType(this.mangledSource, demangled, "undefined");
        function.setReturnType(defaultReturnType);
    }

    private LambdaName getLambdaName(String name) {
        if (!name.startsWith("{")) {
            return null;
        }
        LambdaReplacedString replacedString = new LambdaReplacedString(this, name);
        String updatedName = replacedString.getModifiedText();
        Matcher matcher = LAMBDA_PATTERN.matcher(updatedName);
        if (!matcher.matches()) {
            return null;
        }
        String fullText = matcher.group(1);
        fullText = replacedString.restoreReplacedText(fullText);
        String params = matcher.group(2);
        params = replacedString.restoreReplacedText(params);
        String trailing = matcher.group(3);
        trailing = replacedString.restoreReplacedText(trailing);
        String modifiers = matcher.group(4);
        return new LambdaName(this, fullText, params, trailing, modifiers);
    }

    private String stripOffTemplates(String string) {
        StringBuilder buffy = new StringBuilder();
        int depth = 0;
        for (int i = 0; i < string.length(); ++i) {
            char c = string.charAt(i);
            if (c == '<') {
                ++depth;
                continue;
            }
            if (c == '>') {
                --depth;
                continue;
            }
            if (depth != 0) continue;
            buffy.append(c);
        }
        return buffy.toString();
    }

    private DemangledObject parseItemInNamespace(String itemText) {
        int pos = itemText.lastIndexOf("::");
        if (pos == -1) {
            throw new DemanglerParseException("Expected the demangled string to contain a namespace");
        }
        String parentText = itemText.substring(0, pos);
        DemangledObject parent = this.parseFunctionOrVariable(parentText);
        String name = itemText.substring(pos + 2);
        DemangledObject item = this.parseFunctionOrVariable(name);
        DemangledType namespaceType = this.createNamespaceDemangledType((Demangled)parent);
        item.setNamespace((Demangled)namespaceType);
        return item;
    }

    private String removeBadSpaces(String text) {
        CondensedString condensedString = new CondensedString(this, text);
        return condensedString.getCondensedText();
    }

    private String removeTrailingDereferenceCharacters(String text) {
        char c;
        int i;
        for (i = text.length() - 1; i >= 0 && ((c = text.charAt(i)) == '*' || c == '&'); --i) {
        }
        return text.substring(0, i + 1);
    }

    private List<DemangledParameter> parseParameters(String parameterString) {
        List<String> parameterStrings = this.tokenizeParameters(parameterString);
        List<DemangledParameter> parameters = this.convertIntoParameters(parameterStrings);
        return parameters;
    }

    private List<String> tokenizeParameters(String parameterString) {
        ArrayList<String> parameters = new ArrayList<String>();
        if (parameterString.length() == 0) {
            return parameters;
        }
        Matcher matcher = UNNECESSARY_PARENS_PATTERN.matcher(parameterString);
        if (matcher.matches()) {
            parameterString = matcher.group(1);
        }
        if (StringUtils.isBlank((CharSequence)parameterString)) {
            return parameters;
        }
        int depth = 0;
        int startIndex = 0;
        for (int i = 0; i < parameterString.length(); ++i) {
            int start;
            int end;
            int start2;
            char ch = parameterString.charAt(i);
            if (ch == ',' && depth == 0) {
                String ps = parameterString.substring(startIndex, i);
                parameters.add(ps.trim());
                startIndex = i + 1;
                continue;
            }
            if (ch == '<') {
                ++depth;
                continue;
            }
            if (ch == '>') {
                --depth;
                continue;
            }
            if (ch != '(') continue;
            matcher = ARRAY_POINTER_REFERENCE_PIECE_PATTERN.matcher(parameterString.substring(i));
            if (matcher.find() && (start2 = matcher.start()) == 0) {
                end = matcher.end() - 1;
                i += end;
                continue;
            }
            matcher = CAST_PATTERN.matcher(parameterString.substring(i));
            if (matcher.find() && (start = matcher.start()) == 0) {
                end = matcher.end() - 1;
                i += end;
                continue;
            }
            int end2 = this.findBalancedEnd(parameterString, i, '(', ')');
            if (end2 == -1) {
                end2 = parameterString.length();
            }
            i = end2;
        }
        if (startIndex < parameterString.length()) {
            String ps = parameterString.substring(startIndex, parameterString.length());
            parameters.add(ps.trim());
        }
        return parameters;
    }

    private List<DemangledParameter> convertIntoParameters(List<String> parameterStrings) {
        ArrayList<DemangledParameter> parameters = new ArrayList<DemangledParameter>();
        for (String parameter : parameterStrings) {
            DemangledDataType dt = this.parseParameter(parameter);
            parameters.add(new DemangledParameter(dt));
        }
        return parameters;
    }

    private DemangledDataType parseParameter(String parameter) {
        Matcher castMatcher = CAST_PATTERN.matcher(parameter);
        if (castMatcher.matches()) {
            return new DemangledDataType(this.mangledSource, this.demangledSource, parameter);
        }
        Matcher matcher = VARARGS_IN_PARENS.matcher(parameter);
        if (matcher.matches()) {
            String inside = matcher.group(1);
            DemangledDataType dt = this.parseDataType(inside);
            dt.setVarArgs();
            return dt;
        }
        if ("".equals(parameter.trim())) {
            return new DemangledDataType(this.mangledSource, this.demangledSource, "missing_argument");
        }
        return this.parseDataType(parameter);
    }

    private DemangledDataType parseReturnType(String returnType) {
        return this.parseDataType(returnType);
    }

    private DemangledDataType parseDataType(String fullDatatype) {
        DemangledDataType dt = this.createTypeInNamespace(fullDatatype);
        String datatype = dt.getDemangledName();
        if (this.isMemberPointerOrReference(fullDatatype, datatype)) {
            return this.createMemberPointer(fullDatatype);
        }
        if (this.isLiteral(fullDatatype)) {
            return this.createLiteral(fullDatatype);
        }
        boolean finishedName = false;
        for (int i = 0; i < datatype.length(); ++i) {
            char ch = datatype.charAt(i);
            if (ch == '&' && i == 0 || !finishedName && this.isDataTypeNameCharacter(ch)) continue;
            if (!finishedName) {
                finishedName = true;
                if (VAR_ARGS.equals(datatype)) {
                    dt.setVarArgs();
                } else {
                    String name = datatype.substring(0, i).trim();
                    dt.setName(name);
                }
            }
            if (ch == ' ') continue;
            if (ch == '<') {
                int contentStart = i + 1;
                int templateEnd = this.findTemplateEnd(datatype, i);
                if (templateEnd == -1 || templateEnd > datatype.length()) {
                    throw new DemanglerParseException("Did not find ending to template");
                }
                String templateContent = datatype.substring(contentStart, templateEnd);
                DemangledTemplate template = this.parseTemplate(templateContent);
                dt.setTemplate(template);
                i = templateEnd;
            } else if (ch == '(') {
                LambdaName lambdaName = this.getLambdaName(datatype);
                DemangledDataType newDt = this.tryToParseArrayPointerOrReference(dt, datatype);
                if (newDt != null) {
                    dt = newDt;
                    i = datatype.length();
                } else if (lambdaName != null) {
                    DemangledDataType lambdaArrayDt = this.tryToParseLambdaArrayPointerOrReference(lambdaName, dt, datatype);
                    if (lambdaArrayDt != null) {
                        dt = lambdaArrayDt;
                        i = datatype.length();
                    } else {
                        String fullText = lambdaName.getFullText();
                        dt.setName(fullText);
                        int offset = fullText.indexOf(40);
                        int remaining = fullText.length() - offset;
                        i += remaining;
                        --i;
                    }
                } else {
                    boolean hasPointerParens = this.hasConsecutiveSetsOfParens(datatype.substring(i));
                    if (hasPointerParens) {
                        namespace = dt.getNamespace();
                        DemangledFunctionPointer dfp = this.parseFunctionPointer(datatype);
                        int paramParenEnd = datatype.lastIndexOf(41);
                        if (paramParenEnd == -1) {
                            throw new DemanglerParseException("Did not find ending to closure: " + datatype);
                        }
                        dfp.getReturnType().setNamespace(namespace);
                        dt = dfp;
                        i = paramParenEnd + 1;
                    } else {
                        namespace = dt.getNamespace();
                        DemangledFunctionPointer dfp = this.parseFunction(datatype, i);
                        int firstParenEnd = datatype.indexOf(41, i + 1);
                        if (firstParenEnd == -1) {
                            throw new DemanglerParseException("Did not find ending to closure: " + datatype);
                        }
                        dfp.getReturnType().setNamespace(namespace);
                        dt = dfp;
                        i = firstParenEnd + 1;
                    }
                }
            } else {
                if (ch == '*') {
                    dt.incrementPointerLevels();
                    continue;
                }
                if (ch == '&') {
                    if (!dt.isReference()) {
                        dt.setLValueReference();
                        continue;
                    }
                    dt.setRValueReference();
                    continue;
                }
                if (ch == '[') {
                    dt.setArray(dt.getArrayDimensions() + 1);
                    i = datatype.indexOf(93, i + 1);
                    continue;
                }
            }
            String substr = datatype.substring(i);
            if (substr.startsWith("const")) {
                dt.setConst();
                i += 4;
                continue;
            }
            if (substr.startsWith("struct")) {
                dt.setStruct();
                i += 5;
                continue;
            }
            if (substr.startsWith("class")) {
                dt.setClass();
                i += 4;
                continue;
            }
            if (substr.startsWith("enum")) {
                dt.setEnum();
                i += 3;
                continue;
            }
            if (dt.getName().equals("long")) {
                if (substr.startsWith("long")) {
                    dt.setName("long long");
                    i += 3;
                    continue;
                }
                if (!substr.startsWith("double")) continue;
                dt.setName("long double");
                i += 5;
                continue;
            }
            if (!dt.getName().equals("unsigned")) continue;
            dt.setUnsigned();
            if (substr.startsWith("long")) {
                dt.setName("long");
                i += 3;
                continue;
            }
            if (substr.startsWith("int")) {
                dt.setName("int");
                i += 2;
                continue;
            }
            if (substr.startsWith("short")) {
                dt.setName("short");
                i += 4;
                continue;
            }
            if (!substr.startsWith("char")) continue;
            dt.setName("char");
            i += 3;
        }
        return dt;
    }

    private DemangledDataType createLiteral(String datatype) {
        char lastChar = datatype.charAt(datatype.length() - 1);
        if (lastChar == 'l') {
            return new DemangledDataType(this.mangledSource, this.demangledSource, "long");
        }
        return new DemangledDataType(this.mangledSource, this.demangledSource, "int");
    }

    private boolean isLiteral(String fullDatatype) {
        Matcher m = LITERAL_NUMBER_PATTERN.matcher(fullDatatype);
        return m.matches();
    }

    private DemangledDataType tryToParseLambdaArrayPointerOrReference(LambdaName lambdaName, DemangledDataType dt, String datatype) {
        String fullText = lambdaName.getFullText();
        CustomReplacedString lambdaString = new CustomReplacedString(this, datatype, fullText);
        String noLambdaString = ((ReplacedString)lambdaString).getModifiedText();
        Matcher matcher = ARRAY_POINTER_REFERENCE_PATTERN.matcher(noLambdaString);
        if (!matcher.find()) {
            return null;
        }
        int start = matcher.start(0);
        String leading = noLambdaString.substring(0, start);
        leading = this.removeTrailingDereferenceCharacters(leading);
        Demangled namespace = dt.getNamespace();
        String name = leading;
        DemangledDataType newDt = this.parseArrayPointerOrReference(datatype, name, lambdaString, matcher);
        newDt.setNamespace(namespace);
        return newDt;
    }

    private DemangledDataType tryToParseArrayPointerOrReference(DemangledDataType dt, String datatype) {
        TemplatedString templatedString = new TemplatedString(this, datatype);
        String untemplatedDatatype = ((ReplacedString)templatedString).getModifiedText();
        Matcher matcher = ARRAY_POINTER_REFERENCE_PATTERN.matcher(untemplatedDatatype);
        if (!matcher.find()) {
            return null;
        }
        int start = matcher.start(0);
        String leading = untemplatedDatatype.substring(0, start);
        leading = this.removeTrailingDereferenceCharacters(leading);
        Demangled namespace = dt.getNamespace();
        String name = leading;
        DemangledDataType newDt = this.parseArrayPointerOrReference(datatype, name, templatedString, matcher);
        newDt.setNamespace(namespace);
        return newDt;
    }

    private boolean isMemberPointerOrReference(String fullDataType, String datatype) {
        String test = datatype;
        if (!(test = test.replaceAll("const|\\*|&|\\s", "")).isEmpty()) {
            return false;
        }
        return fullDataType.endsWith("::" + datatype);
    }

    private boolean hasConsecutiveSetsOfParens(String text) {
        int end = this.findBalancedEnd(text, 0, '(', ')');
        if (end < -1) {
            return false;
        }
        String remaining = text.substring(end + 1).trim();
        return remaining.startsWith("(");
    }

    private DemangledDataType createMemberPointer(String datatype) {
        DemangledDataType dt;
        int namespaceEnd = datatype.lastIndexOf("::");
        String typeWithoutPointer = datatype.substring(0, namespaceEnd);
        int space = typeWithoutPointer.indexOf(32);
        if (space != -1) {
            String type = typeWithoutPointer.substring(0, space);
            dt = this.createTypeInNamespace(type);
            String parentType = typeWithoutPointer.substring(space + 1);
            DemangledDataType parentDt = this.createTypeInNamespace(parentType);
            dt.setNamespace((Demangled)parentDt);
        } else {
            dt = this.createTypeInNamespace(typeWithoutPointer);
        }
        dt.incrementPointerLevels();
        return dt;
    }

    private boolean isDataTypeNameCharacter(char ch) {
        return Character.isLetter(ch) || Character.isDigit(ch) || ch == ':' || ch == '_' || ch == '{' || ch == '$';
    }

    private int findBalancedEnd(String string, int start, char open, char close) {
        boolean found = false;
        int depth = 0;
        for (int i = start; i < string.length(); ++i) {
            char c = string.charAt(i);
            if (c == open) {
                ++depth;
                found = true;
            } else if (c == close) {
                --depth;
            }
            if (!found || depth != 0) continue;
            return i;
        }
        return -1;
    }

    private int findBalancedStart(String string, int start, char open, char close) {
        boolean found = false;
        int depth = 0;
        for (int i = start; i >= 0; --i) {
            char c = string.charAt(i);
            if (c == open) {
                --depth;
            } else if (c == close) {
                ++depth;
                found = true;
            }
            if (!found || depth != 0) continue;
            return i;
        }
        return -1;
    }

    private int findTemplateEnd(String string, int start) {
        return this.findBalancedEnd(string, start, '<', '>');
    }

    private int findTemplateStart(String string, int templateEnd) {
        return this.findBalancedStart(string, templateEnd, '<', '>');
    }

    private int findNamespaceStart(String text, int start, int stop) {
        if (!text.contains("::")) {
            return -1;
        }
        int colonCount = 0;
        int parenDepth = 0;
        int templateDepth = 0;
        int braceDepth = 0;
        boolean isNested = false;
        block10: for (int i = start; i >= stop; --i) {
            char c = text.charAt(i);
            switch (c) {
                case ':': {
                    if (++colonCount != 2) continue block10;
                    if (!isNested) {
                        return i + 2;
                    }
                    colonCount = 0;
                    continue block10;
                }
                case ' ': {
                    if (isNested) continue block10;
                    return -1;
                }
                case '(': {
                    isNested = --parenDepth > 0 || templateDepth > 0 || braceDepth > 0;
                    continue block10;
                }
                case ')': {
                    isNested = ++parenDepth > 0 || templateDepth > 0 || braceDepth > 0;
                    continue block10;
                }
                case '<': {
                    isNested = parenDepth > 0 || --templateDepth > 0 || braceDepth > 0;
                    continue block10;
                }
                case '>': {
                    isNested = parenDepth > 0 || ++templateDepth > 0 || braceDepth > 0;
                    continue block10;
                }
                case '{': {
                    isNested = parenDepth > 0 || templateDepth > 0 || --braceDepth > 0;
                    continue block10;
                }
                case '}': {
                    isNested = parenDepth > 0 || templateDepth > 0 || ++braceDepth > 0;
                    continue block10;
                }
                default: {
                    continue block10;
                }
            }
        }
        return -1;
    }

    private DemangledDataType createTypeInNamespace(String name) {
        List names = SymbolPathParser.parse((String)name, (boolean)false);
        DemangledType namespace = null;
        if (names.size() > 1) {
            namespace = this.convertToNamespaces(names.subList(0, names.size() - 1));
        }
        String datatypeName = (String)names.get(names.size() - 1);
        DemangledDataType dt = new DemangledDataType(this.mangledSource, this.demangledSource, datatypeName);
        dt.setName(datatypeName);
        dt.setNamespace((Demangled)namespace);
        return dt;
    }

    private void setNameAndNamespace(DemangledObject object, String name) {
        List names = SymbolPathParser.parse((String)name, (boolean)false);
        DemangledType namespace = null;
        if (names.size() > 1) {
            namespace = this.convertToNamespaces(names.subList(0, names.size() - 1));
        }
        String objectName = (String)names.get(names.size() - 1);
        object.setName(objectName);
        object.setNamespace((Demangled)namespace);
    }

    private void setNamespace(DemangledObject object, String name) {
        List names = SymbolPathParser.parse((String)name, (boolean)false);
        object.setNamespace((Demangled)this.convertToNamespaces(names));
    }

    private DemangledTemplate parseTemplate(String string) {
        String contents = string;
        if (string.startsWith("<") && string.endsWith(">")) {
            contents = string.substring(1, string.length() - 1);
        }
        List<DemangledParameter> parameters = this.parseParameters(contents);
        DemangledTemplate template = new DemangledTemplate();
        for (DemangledParameter parameter : parameters) {
            template.addParameter(parameter.getType());
        }
        return template;
    }

    private DemangledDataType parseArrayPointerOrReference(String datatype, String name, ReplacedString replacedString, Matcher matcher) {
        String realName = replacedString.restoreReplacedText(name);
        DemangledDataType dt = new DemangledDataType(this.mangledSource, this.demangledSource, realName);
        String type = matcher.group(1);
        if (type.equals("*")) {
            dt.incrementPointerLevels();
        } else if (type.equals("&")) {
            dt.setLValueReference();
        } else {
            throw new DemanglerParseException("Unexpected charater inside of parens: " + type);
        }
        String safeDatatype = replacedString.getModifiedText();
        int midTextStart = safeDatatype.indexOf(name) + name.length();
        int midTextEnd = matcher.start(1) - 1;
        String midText = safeDatatype.substring(midTextStart, midTextEnd);
        if (midText.contains(CONST)) {
            dt.setConst();
        }
        int pointers = StringUtils.countMatches((CharSequence)midText, (char)'*');
        for (int i = 0; i < pointers; ++i) {
            dt.incrementPointerLevels();
        }
        String arraySubscripts = matcher.group(2);
        int arrays = StringUtilities.countOccurrences((String)arraySubscripts, (char)'[');
        dt.setArray(arrays);
        return dt;
    }

    private DemangledFunctionPointer parseFunctionPointer(String functionString) {
        int parenStart = functionString.indexOf(40);
        int parenEnd = this.findBalancedEnd(functionString, parenStart, '(', ')');
        String returnType = functionString.substring(0, parenStart).trim();
        int paramStart = functionString.indexOf(40, parenEnd + 1);
        int paramEnd = functionString.lastIndexOf(41);
        String parameters = functionString.substring(paramStart + 1, paramEnd);
        DemangledFunctionPointer dfp = this.createFunctionPointer(parameters, returnType);
        return dfp;
    }

    private DemangledFunctionPointer parseFunction(String functionString, int offset) {
        int parenStart = functionString.indexOf(40, offset);
        int parenEnd = this.findBalancedEnd(functionString, parenStart, '(', ')');
        String returnType = functionString.substring(0, parenStart).trim();
        int paramStart = parenStart;
        int paramEnd = parenEnd;
        String parameters = functionString.substring(paramStart + 1, paramEnd);
        DemangledFunctionPointer dfp = this.createFunctionPointer(parameters, returnType);
        dfp.setDisplayDefaultFunctionPointerSyntax(false);
        return dfp;
    }

    private DemangledFunctionPointer createFunctionPointer(String parameterString, String returnType) {
        List<DemangledParameter> parameters = this.parseParameters(parameterString);
        DemangledFunctionPointer dfp = new DemangledFunctionPointer(this.mangledSource, this.demangledSource);
        DemangledDataType returnDataType = this.parseReturnType(returnType);
        dfp.setReturnType(returnDataType);
        for (DemangledParameter parameter : parameters) {
            dfp.addParameter(parameter.getType());
        }
        return dfp;
    }

    private DemangledObject parseVariable(String demangled) {
        String nameString = this.removeBadSpaces(demangled).trim();
        DemangledVariable variable = new DemangledVariable(this.mangledSource, this.demangledSource, (String)null);
        this.setNameAndNamespace((DemangledObject)variable, nameString);
        return variable;
    }

    private DemangledType convertToNamespaces(List<String> names) {
        DemangledType myNamespace;
        if (names.size() == 0) {
            return null;
        }
        int index = names.size() - 1;
        String rawName = names.get(index);
        String escapedName = this.removeBadSpaces(rawName);
        DemangledType namespace = myNamespace = new DemangledType(this.mangledSource, this.demangledSource, escapedName);
        while (--index >= 0) {
            rawName = names.get(index);
            escapedName = this.removeBadSpaces(rawName);
            DemangledType parentNamespace = new DemangledType(this.mangledSource, this.demangledSource, escapedName);
            namespace.setNamespace((Demangled)parentNamespace);
            namespace = parentNamespace;
        }
        return myNamespace;
    }

    private static List<GnuDemanglerReplacement> loadStandardReplacements() {
        ArrayList<GnuDemanglerReplacement> results = new ArrayList<GnuDemanglerReplacement>();
        List files = Application.findFilesByExtensionInApplication((String)REPLACEMENT_FILE_EXTENSION);
        files.sort((r1, r2) -> r1.getName().compareTo(r2.getName()));
        for (ResourceFile file : files) {
            List<String> lines = GnuDemanglerParser.getReplacementLines(file);
            for (String line : lines) {
                GnuDemanglerReplacement replacement = GnuDemanglerParser.parseReplacement(line, file);
                if (replacement == null) continue;
                results.add(replacement);
            }
        }
        return results;
    }

    private static GnuDemanglerReplacement parseReplacement(String line, ResourceFile file) {
        for (int i = 0; i < line.length(); ++i) {
            char c = line.charAt(i);
            if (!Character.isWhitespace(c)) continue;
            String replace = line.substring(0, i).trim();
            if (replace.equals("\"\"")) {
                replace = "";
            }
            String find = line.substring(i).trim();
            return new GnuDemanglerReplacement(find, replace, file);
        }
        Msg.warn(GnuDemanglerParser.class, (Object)("Malformed replacement line.  No spaces found: " + line + ". In file: " + String.valueOf(file)));
        return null;
    }

    private static List<String> getReplacementLines(ResourceFile file) {
        try {
            List lines = FileUtilities.getLines((ResourceFile)file);
            return lines.stream().filter(l -> !l.startsWith("//")).filter(l -> !l.isBlank()).map(String::trim).collect(Collectors.toList());
        }
        catch (IOException e) {
            Msg.error(GnuDemanglerParser.class, (Object)("Unable to load gnu demangler replacement file: " + String.valueOf(file)), (Throwable)e);
            return Collections.emptyList();
        }
    }

    private DemangledType createNamespaceDemangledType(Demangled namespace) {
        String namespaceName = namespace.getNamespaceName();
        String escapedName = this.removeBadSpaces(namespaceName);
        DemangledType type = new DemangledType(this.mangledSource, this.demangledSource, escapedName);
        type.setNamespace(namespace.getNamespace());
        return type;
    }

    private boolean nextCharIs(String text, int index, char c) {
        char next = text.charAt(index);
        while (next == ' ') {
            next = text.charAt(++index);
        }
        return next == c;
    }

    private abstract class DemangledObjectBuilder {
        protected String demangled;

        DemangledObjectBuilder(GnuDemanglerParser gnuDemanglerParser, String demangled) {
            this.demangled = demangled;
        }

        abstract DemangledObject build();
    }

    private abstract class SpecialPrefixHandler
    extends DemangledObjectBuilder {
        protected String prefix;
        protected String name;
        protected String type;

        SpecialPrefixHandler(String demangled) {
            super(GnuDemanglerParser.this, demangled);
        }

        @Override
        DemangledObject build() {
            DemangledObject dobj = GnuDemanglerParser.this.parseFunctionOrVariable(this.type);
            return this.doBuild((Demangled)dobj);
        }

        abstract DemangledObject doBuild(Demangled var1);

        public String toString() {
            ToStringBuilder builder = new ToStringBuilder((Object)this, ToStringStyle.JSON_STYLE);
            return builder.append("name", (Object)this.name).append("prefix", (Object)this.prefix).append("type", (Object)this.type).append("demangled", (Object)this.demangled).toString();
        }
    }

    private abstract class OperatorHandler
    extends DemangledObjectBuilder {
        protected Matcher matcher;

        OperatorHandler(GnuDemanglerParser gnuDemanglerParser, String demangled) {
            super(gnuDemanglerParser, demangled);
        }

        abstract boolean matches(String var1);
    }

    private class ItemInNamespaceHandler
    extends SpecialPrefixHandler {
        ItemInNamespaceHandler(String demangled) {
            super(demangled);
            this.type = demangled;
        }

        ItemInNamespaceHandler(String demangled, String prefix, String item) {
            super(demangled);
            this.prefix = prefix;
            this.type = item;
        }

        @Override
        DemangledObject doBuild(Demangled namespace) {
            DemangledObject demangledObject = GnuDemanglerParser.this.parseItemInNamespace(this.type);
            return demangledObject;
        }
    }

    private class OverloadOperatorHandler
    extends OperatorHandler {
        OverloadOperatorHandler(String demangled) {
            super(GnuDemanglerParser.this, demangled);
        }

        @Override
        boolean matches(String text) {
            this.matcher = OVERLOAD_OPERATOR_NAME_PATTERN.matcher(text);
            if (!this.matcher.matches()) {
                return false;
            }
            int operatorStart = this.matcher.start(2);
            int leafStart = GnuDemanglerParser.this.findNamespaceStart(this.demangled, text.length() - 1, operatorStart);
            return leafStart <= operatorStart;
        }

        @Override
        DemangledObject build() {
            String operatorChars = this.matcher.group(2);
            int start = this.matcher.start(2);
            int end = this.matcher.end(2);
            String templates = this.getTemplates(end);
            String placeholder = "TEMPNAMEPLACEHOLDERVALUE";
            String baseOperator = GnuDemanglerParser.OPERATOR + this.demangled.substring(start, end += templates.length());
            String fixedFunction = this.demangled.replace(baseOperator, placeholder);
            DemangledFunction function = (DemangledFunction)GnuDemanglerParser.this.parseFunctionOrVariable(fixedFunction);
            function.setOverloadedOperator(true);
            String simpleName = GnuDemanglerParser.OPERATOR + operatorChars;
            if (StringUtils.isBlank((CharSequence)templates)) {
                function.setName(simpleName);
            } else {
                String escapedTemplates = GnuDemanglerParser.this.removeBadSpaces(templates);
                DemangledTemplate demangledTemplate = GnuDemanglerParser.this.parseTemplate(escapedTemplates);
                function.setTemplate(demangledTemplate);
                function.setName(simpleName);
            }
            return function;
        }

        private String getTemplates(int start) {
            String templates = "";
            boolean hasTemplates = GnuDemanglerParser.this.nextCharIs(this.demangled, start, '<');
            if (hasTemplates) {
                int templateStart = start;
                int templateEnd = GnuDemanglerParser.this.findTemplateEnd(this.demangled, templateStart);
                if (templateEnd == -1) {
                    Msg.debug((Object)this, (Object)("Unable to find template end for operator: " + this.demangled));
                    return templates;
                }
                templates = this.demangled.substring(templateStart, templateEnd + 1);
            }
            return templates;
        }
    }

    private class ConversionOperatorHandler
    extends OperatorHandler {
        ConversionOperatorHandler(String demangled) {
            super(GnuDemanglerParser.this, demangled);
        }

        @Override
        boolean matches(String text) {
            this.matcher = CONVERSION_OPERATOR_PATTERN.matcher(text);
            return this.matcher.matches();
        }

        @Override
        DemangledObject build() {
            String fullName = this.matcher.group(1);
            String fullReturnType = this.matcher.group(2);
            boolean isConst = false;
            int index = fullReturnType.indexOf(GnuDemanglerParser.CONST);
            if (index != -1) {
                fullReturnType = fullReturnType.replace(GnuDemanglerParser.CONST, "");
                isConst = true;
            }
            DemangledFunction method = new DemangledFunction(GnuDemanglerParser.this.mangledSource, GnuDemanglerParser.this.demangledSource, (String)null);
            DemangledDataType returnType = GnuDemanglerParser.this.parseReturnType(fullReturnType);
            if (isConst) {
                returnType.setConst();
            }
            method.setReturnType(returnType);
            int operatorIndex = fullName.lastIndexOf("::operator");
            String namespace = fullName.substring(0, operatorIndex);
            String templatelessNamespace = GnuDemanglerParser.this.stripOffTemplates(namespace);
            GnuDemanglerParser.this.setNamespace((DemangledObject)method, templatelessNamespace);
            String templatelessReturnType = GnuDemanglerParser.this.stripOffTemplates(fullReturnType);
            List path = SymbolPathParser.parse((String)templatelessReturnType, (boolean)false);
            String shortReturnTypeName = (String)path.get(path.size() - 1);
            if (shortReturnTypeName.contains("(")) {
                shortReturnTypeName = "function.pointer";
            }
            method.setName("operator.cast.to." + shortReturnTypeName);
            method.setBackupPlateComment(fullName + " " + fullReturnType + "()");
            method.setOverloadedOperator(true);
            return method;
        }
    }

    private class NewOrDeleteOperatorHandler
    extends OperatorHandler {
        NewOrDeleteOperatorHandler(String demangled) {
            super(GnuDemanglerParser.this, demangled);
        }

        @Override
        boolean matches(String demangler) {
            this.matcher = NEW_DELETE_OPERATOR_PATTERN.matcher(demangler);
            return this.matcher.matches();
        }

        @Override
        DemangledObject build() {
            String operatorText = this.matcher.group(1);
            String operatorName = this.matcher.group(2);
            String arrayBrackets = this.matcher.group(3);
            String parametersText = this.matcher.group(4);
            DemangledFunction function = new DemangledFunction(GnuDemanglerParser.this.mangledSource, GnuDemanglerParser.this.demangledSource, (String)null);
            function.setOverloadedOperator(true);
            DemangledDataType returnType = new DemangledDataType(GnuDemanglerParser.this.mangledSource, GnuDemanglerParser.this.demangledSource, "void");
            if (operatorName.startsWith("new")) {
                returnType.incrementPointerLevels();
            }
            function.setReturnType(returnType);
            GnuDemanglerParser.this.setNameAndNamespace((DemangledObject)function, operatorText);
            List<DemangledParameter> parameters = GnuDemanglerParser.this.parseParameters(parametersText);
            for (DemangledParameter parameter : parameters) {
                function.addParameter(parameter);
            }
            Object name = operatorName;
            if (arrayBrackets != null) {
                name = (String)name + "[]";
            }
            function.setName("operator." + (String)name);
            function.setBackupPlateComment(operatorText + " " + operatorName);
            return function;
        }
    }

    private class ThunkHandler
    extends SpecialPrefixHandler {
        ThunkHandler(String demangled, String prefix, String item) {
            super(demangled);
            this.demangled = demangled;
            this.prefix = prefix;
            this.type = item;
        }

        @Override
        DemangledObject doBuild(Demangled demangledObject) {
            DemangledFunction function = (DemangledFunction)demangledObject;
            function.setCallingConvention("__thiscall");
            DemangledThunk thunk = new DemangledThunk(GnuDemanglerParser.this.mangledSource, GnuDemanglerParser.this.demangledSource, function);
            if (this.prefix.contains(GnuDemanglerParser.COVARIANT_RETURN_THUNK)) {
                thunk.setCovariantReturnThunk();
            }
            thunk.setSignaturePrefix(this.prefix);
            return thunk;
        }
    }

    private class AddressTableHandler
    extends SpecialPrefixHandler {
        AddressTableHandler(String demangled, String prefix, String type) {
            super(demangled);
            this.demangled = demangled;
            this.prefix = prefix;
            this.type = type;
            int pos = prefix.trim().lastIndexOf(32);
            this.name = prefix.substring(0, pos).replace(' ', '-');
        }

        @Override
        DemangledObject doBuild(Demangled namespace) {
            DemangledAddressTable addressTable = new DemangledAddressTable(GnuDemanglerParser.this.mangledSource, this.demangled, this.name, true);
            DemangledType namespaceType = GnuDemanglerParser.this.createNamespaceDemangledType(namespace);
            addressTable.setNamespace((Demangled)namespaceType);
            return addressTable;
        }
    }

    private class TypeInfoNameHandler
    extends SpecialPrefixHandler {
        TypeInfoNameHandler(String demangled, String prefix) {
            String classname;
            super(demangled);
            this.demangled = demangled;
            this.prefix = prefix;
            this.type = classname = demangled.substring(prefix.length()).trim();
        }

        @Override
        DemangledObject doBuild(Demangled namespace) {
            DemangledString demangledString = new DemangledString(GnuDemanglerParser.this.mangledSource, GnuDemanglerParser.this.demangledSource, "typeinfo-name", this.type, -1, false);
            demangledString.setSpecialPrefix(GnuDemanglerParser.TYPEINFO_NAME_FOR);
            String namespaceString = GnuDemanglerParser.this.removeBadSpaces(this.type);
            GnuDemanglerParser.this.setNamespace((DemangledObject)demangledString, namespaceString);
            return demangledString;
        }
    }

    private class ArrayHandler
    extends SpecialPrefixHandler {
        private String arrayType;

        ArrayHandler(String demangled, String prefix, String item) {
            super(demangled);
            this.demangled = demangled;
            this.prefix = prefix;
            this.type = item;
            Matcher arrayMatcher = ARRAY_DATA_PATTERN.matcher(this.type);
            if (arrayMatcher.matches()) {
                this.type = arrayMatcher.group(1);
                this.arrayType = arrayMatcher.group(2).trim();
            }
        }

        @Override
        DemangledObject doBuild(Demangled namespace) {
            DemangledObject demangledObject = GnuDemanglerParser.this.parseItemInNamespace(this.type);
            if (demangledObject instanceof DemangledVariable) {
                DemangledVariable variable = (DemangledVariable)demangledObject;
                if ("char".equals(this.arrayType) && this.type.contains("StringLiteral")) {
                    DemangledString ds = new DemangledString(variable.getMangledString(), this.demangled, this.type, this.type, -1, false);
                    ds.setSpecialPrefix(this.prefix);
                    return ds;
                }
            }
            return demangledObject;
        }
    }

    private class GlobalCtorDtorHandler
    extends SpecialPrefixHandler {
        GlobalCtorDtorHandler(String demangled, String prefix, String item) {
            super(demangled);
            this.prefix = prefix;
            this.type = item;
        }

        @Override
        DemangledObject doBuild(Demangled namespace) {
            Object functionName = this.type;
            if (!((String)functionName).contains("(")) {
                functionName = (String)functionName + "()";
            }
            DemangledObject demangledFunction = GnuDemanglerParser.this.parseFunctionOrVariable((String)functionName);
            demangledFunction.setOriginalDemangled(this.demangled);
            String parsedFunctionName = demangledFunction.getName();
            String prefixNoSpaces = this.prefix.replaceAll(" ", "\\.");
            String fullName = prefixNoSpaces + parsedFunctionName;
            demangledFunction.setName(fullName);
            demangledFunction.setBackupPlateComment(this.demangled);
            return demangledFunction;
        }
    }

    private class FunctionSignatureParts {
        private boolean isFunction;
        private String returnType;
        private String name;
        private String rawParameterPrefix;
        private List<DemangledParameter> parameters;

        FunctionSignatureParts(GnuDemanglerParser gnuDemanglerParser, String signatureString) {
            ParameterLocator paramLocator = new ParameterLocator(gnuDemanglerParser, signatureString);
            if (!paramLocator.hasParameters()) {
                return;
            }
            this.isFunction = true;
            int paramStart = paramLocator.getParamStart();
            int paramEnd = paramLocator.getParamEnd();
            String parameterString = signatureString.substring(paramStart + 1, paramEnd).trim();
            this.parameters = gnuDemanglerParser.parseParameters(parameterString);
            int prefixEndPos = paramStart;
            this.rawParameterPrefix = signatureString.substring(0, prefixEndPos).trim();
            CondensedString prefixString = new CondensedString(gnuDemanglerParser, this.rawParameterPrefix);
            String prefix = prefixString.getCondensedText();
            int nameStart = Math.max(0, prefix.lastIndexOf(32));
            this.name = prefix.substring(nameStart, prefix.length()).trim();
            if (nameStart > 0) {
                this.returnType = prefix.substring(0, nameStart);
            }
        }

        String getReturnType() {
            return this.returnType;
        }

        String getName() {
            return this.name;
        }

        String getRawParameterPrefix() {
            return this.rawParameterPrefix;
        }

        boolean isValidFunction() {
            return this.isFunction;
        }

        List<DemangledParameter> getParameters() {
            return this.parameters;
        }

        public String toString() {
            return Json.toString((Object)this);
        }
    }

    private class LambdaName {
        private String fullText;
        private String params;
        private String id;
        private String trailing;

        LambdaName(GnuDemanglerParser gnuDemanglerParser, String fullText, String params, String id, String trailing) {
            this.fullText = fullText;
            this.params = params;
            this.id = id;
            this.trailing = trailing == null ? "" : trailing;
        }

        String getFullText() {
            return this.fullText;
        }

        public String toString() {
            ToStringBuilder builder = new ToStringBuilder((Object)this, ToStringStyle.JSON_STYLE);
            return builder.append("fullText", (Object)this.fullText).append("params", (Object)this.params).append("id", (Object)this.id).append("trailing", (Object)this.trailing).toString();
        }
    }

    private class LambdaReplacedString
    extends ReplacedString {
        private String placeholderText = this.getClass().getSimpleName().toUpperCase() + "REPLACEDSTRINGTEMPNAMEPLACEHOLDERVALUE";
        private String modifiedText;

        LambdaReplacedString(GnuDemanglerParser gnuDemanglerParser, String input) {
            super(gnuDemanglerParser, input);
            StringBuilder buffer = new StringBuilder();
            Pattern p = Pattern.compile(GnuDemanglerParser.LAMBDA);
            Matcher matcher = p.matcher(input);
            matcher.find();
            while (matcher.find()) {
                matcher.appendReplacement(buffer, this.placeholderText);
            }
            matcher.appendTail(buffer);
            this.modifiedText = buffer.toString();
        }

        @Override
        String restoreReplacedText(String s) {
            return s.replaceAll(this.placeholderText, GnuDemanglerParser.LAMBDA);
        }

        @Override
        String getModifiedText() {
            return this.modifiedText;
        }
    }

    private class CondensedString {
        private String sourceText;
        private String condensedText;
        private List<Part> parts = new ArrayList<Part>();

        CondensedString(GnuDemanglerParser gnuDemanglerParser, String input) {
            String fixed;
            this.sourceText = fixed = this.fixupUnnamedTypes(input);
            this.condensedText = this.convertTemplateAndParameterSpaces(fixed);
        }

        private String convertTemplateAndParameterSpaces(String name) {
            int depth = 0;
            char last = '\u0000';
            for (int i = 0; i < name.length(); ++i) {
                Part part = new Part(this);
                this.parts.add(part);
                char ch = name.charAt(i);
                part.condensed = part.original = Character.toString(ch);
                if (ch == '<' || ch == '(') {
                    ++depth;
                } else if ((ch == '>' || ch == ')') && depth != 0) {
                    --depth;
                }
                if (depth > 0 && ch == ' ') {
                    char next = i + 1 < name.length() ? name.charAt(i + 1) : (char)'\u0000';
                    part.condensed = this.isSurroundedByCharacters(last, next) ? Character.toString('_') : "";
                }
                last = ch;
            }
            return this.parts.stream().map(p -> p.condensed).collect(Collectors.joining()).trim();
        }

        private boolean isSurroundedByCharacters(char last, char next) {
            if (last == '\u0000' || next == '\u0000') {
                return false;
            }
            return Character.isLetterOrDigit(last) && Character.isLetterOrDigit(next);
        }

        private String fixupUnnamedTypes(String demangled) {
            String fixed = demangled;
            Matcher matcher = UNNAMED_TYPE_PATTERN.matcher(demangled);
            while (matcher.find()) {
                String text = matcher.group(1);
                String noSpace = text.replaceFirst("\\s", "_");
                fixed = fixed.replace(text, noSpace);
            }
            return fixed;
        }

        String getCondensedText() {
            return this.condensedText;
        }

        public String toString() {
            return Json.toString((Object)this);
        }

        private class Part {
            String original;
            String condensed = "";

            private Part(CondensedString condensedString) {
            }

            public String toString() {
                return Json.toString((Object)this);
            }
        }
    }

    private class CustomReplacedString
    extends ReplacedString {
        private String placeholderText = this.getClass().getSimpleName().toUpperCase() + "REPLACEDSTRINGTEMPNAMEPLACEHOLDERVALUE";
        private String replacedText;
        private String modifiedText;

        CustomReplacedString(GnuDemanglerParser gnuDemanglerParser, String input, String textToReplace) {
            super(gnuDemanglerParser, input);
            this.replacedText = textToReplace;
            this.modifiedText = input.replace(textToReplace, this.placeholderText);
        }

        @Override
        String restoreReplacedText(String mutatedText) {
            return mutatedText.replace(this.placeholderText, this.replacedText);
        }

        @Override
        String getModifiedText() {
            return this.modifiedText;
        }
    }

    private abstract class ReplacedString {
        static final String PLACEHOLDER = "REPLACEDSTRINGTEMPNAMEPLACEHOLDERVALUE";
        private String sourceText;

        ReplacedString(GnuDemanglerParser gnuDemanglerParser, String sourceText) {
            this.sourceText = sourceText;
        }

        abstract String restoreReplacedText(String var1);

        abstract String getModifiedText();

        public String toString() {
            return Json.toString((Object)this);
        }
    }

    private class TemplatedString
    extends ReplacedString {
        private String placeholderText = this.getClass().getSimpleName().toUpperCase() + "REPLACEDSTRINGTEMPNAMEPLACEHOLDERVALUE";
        private String replacedText;
        private String modifiedText;

        TemplatedString(GnuDemanglerParser gnuDemanglerParser, String input) {
            super(gnuDemanglerParser, input);
            this.replaceTemplates(input);
        }

        private void replaceTemplates(String string) {
            StringBuilder buffy = new StringBuilder();
            StringBuilder templateBuffer = new StringBuilder();
            int depth = 0;
            for (int i = 0; i < string.length(); ++i) {
                char c = string.charAt(i);
                if (c == '<') {
                    if (depth == 0) {
                        buffy.append(this.placeholderText);
                    }
                    templateBuffer.append(c);
                    ++depth;
                    continue;
                }
                if (c == '>') {
                    templateBuffer.append(c);
                    --depth;
                    continue;
                }
                if (depth == 0) {
                    buffy.append(c);
                    continue;
                }
                templateBuffer.append(c);
            }
            this.modifiedText = buffy.toString();
            this.replacedText = templateBuffer.toString();
        }

        @Override
        String restoreReplacedText(String s) {
            return s.replace(this.placeholderText, this.replacedText);
        }

        @Override
        String getModifiedText() {
            return this.modifiedText;
        }
    }

    private class ParameterLocator {
        int paramStart = -1;
        int paramEnd = -1;
        private String text;

        ParameterLocator(GnuDemanglerParser gnuDemanglerParser, String text) {
            this.text = text;
            this.paramEnd = text.lastIndexOf(41);
            if (this.paramEnd < 0) {
                return;
            }
            if (this.isContainedWithinNamespace()) {
                this.paramEnd = -1;
                return;
            }
            this.paramStart = this.findParameterStart(text, this.paramEnd);
            int templateEnd = gnuDemanglerParser.findTemplateEnd(text, 0);
            int templateStart = -1;
            if (templateEnd != -1) {
                templateStart = gnuDemanglerParser.findTemplateStart(text, templateEnd);
            }
            if (this.paramStart > templateStart && this.paramStart < templateEnd) {
                this.paramStart = -1;
                this.paramEnd = -1;
            }
        }

        public String toString() {
            ToStringBuilder builder = new ToStringBuilder((Object)this, ToStringStyle.JSON_STYLE);
            return builder.append("text", (Object)this.text).append("paramStart", this.paramStart).append("paramEnd", this.paramEnd).toString();
        }

        private boolean isContainedWithinNamespace() {
            return this.paramEnd < this.text.length() - 1 && ':' == this.text.charAt(this.paramEnd + 1);
        }

        int getParamStart() {
            return this.paramStart;
        }

        int getParamEnd() {
            return this.paramEnd;
        }

        boolean hasParameters() {
            return this.paramStart != -1 && this.paramEnd != -1;
        }

        private int findParameterStart(String demangled, int end) {
            int depth = 0;
            for (int i = end - 1; i >= 0; --i) {
                char ch = demangled.charAt(i);
                if (ch == '(' && depth == 0) {
                    return i;
                }
                if (ch == '>' || ch == ')') {
                    ++depth;
                    continue;
                }
                if (ch != '<' && ch != '(') continue;
                --depth;
            }
            return -1;
        }
    }
}

